<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Вихід, якщо доступ до нього прямий
}
/**
 * Simple SEO class to create page metatags: title, description, robots, keywords, Open Graph.
 *
 * @author Kama
 * @version 1
 */
class solass_wp_SEO_Tags {

	function __construct(){}

	static function init(){

		// remove basic title call
		remove_action( 'wp_head', '_wp_render_title_tag', 1 );
		add_action( 'wp_head', [ __CLASS__, 'render_seo_tags' ], 1 );
	}

	static function render_seo_tags(){
		//remove_theme_support( 'title-tag' ); // не обязательно

		echo '<title>'. self::meta_title(' — ') .'</title>'."\n";

		echo self::meta_description();
		echo self::meta_keywords();
		echo self::meta_robots('cpage');

		echo self::og_meta(); // Open Graph, twitter данные
	}

	/**
	 * Open Graph, twitter data in `<head>`.
	 *
	 * See docs: http://ogp.me/
	 *
	 * @version 8
	 */
	static function og_meta(){

		$obj = get_queried_object();

		if( isset($obj->post_type) )   $post = $obj;
		elseif( isset($obj->term_id) ) $term = $obj;

		$is_post = isset( $post );
		$is_term = isset( $term );

		$title = self::meta_title( '–' );
		$desc  = preg_replace( '/^.+content="([^"]*)".*$/s', '$1', self::meta_description() );

		// Open Graph
		$els = [];
		$els['og:locale']      = '<meta property="og:locale" content="'     . get_locale()                           .'" />';
		$els['og:site_name']   = '<meta property="og:site_name" content="'  . esc_attr( get_bloginfo('name') )       .'" />';
		$els['og:title']       = '<meta property="og:title" content="'      . esc_attr( $title )                     .'" />';
		$els['og:description'] = '<meta property="og:description" content="'. esc_attr( $desc )                      .'" />';
		$els['og:type']        = '<meta property="og:type" content="'       .( is_singular() ? 'article' : 'object' ).'" />';

		if( $is_post ) $pageurl = get_permalink( $post );
		if( $is_term ) $pageurl = get_term_link( $term );
		if( isset($pageurl) )
			$els['og:url'] = '<meta property="og:url" content="'. esc_attr( $pageurl ) .'" />';

		/**
		 * Allow to disable `article:section` property.
		 *
		 * @param bool $is_on
		 */
		if( apply_filters( 'kama_og_meta_show_article_section', true ) ){

			if( is_singular() && $post_taxname =  get_object_taxonomies($post->post_type) ){

				$post_terms = get_the_terms( $post, reset($post_taxname) );
				if( $post_terms && $post_term = array_shift($post_terms) )
					$els['article:section'] = '<meta property="article:section" content="'. esc_attr( $post_term->name ) .'" />';
			}
		}

		// image
		if( 'image' ){

			$fn__get_thumb_id_from_text = function( $text ){
				if(
					preg_match( '/<img +src *= *[\'"]([^\'"]+)[\'"]/', $text, $mm ) &&
					( $mm[1]{0} === '/' || strpos($mm[1], $_SERVER['HTTP_HOST']) )
				){
					$name = basename( $mm[1] );
					$name = preg_replace('~-[0-9]+x[0-9]+(?=\..{2,6})~', '', $name ); // удалим размер (-80x80)
					$name = preg_replace('~\.[^.]+$~', '', $name );                   // удалим расширение
					$name = sanitize_title( sanitize_file_name( $name ) );            // приведем к стандартному виду

					global $wpdb;
					$thumb_id = $wpdb->get_var(
						$wpdb->prepare( "SELECT ID FROM $wpdb->posts WHERE post_name = %s AND post_type = 'attachment'", $name )
					);
				}

				return empty($thumb_id) ? 0 : $thumb_id;
			};

			$thumb_id = 0;
			if( $is_post ){

				if( ! $thumb_id = get_post_thumbnail_id( $post ) ){

					/**
					 * Allows to turn off the image search in post content.
					 *
					 * @param bool $is_on
					 */
					if( apply_filters( 'kama_og_meta_thumb_id_find_in_content', true ) ){

						if( ! $thumb_id = $fn__get_thumb_id_from_text( $post->post_content ) ) {
							// первое вложение поста
							$attach = get_children(
								[ 'numberposts'=>'1', 'post_mime_type'=>'image', 'post_type'=>'attachment', 'post_parent'=>$post->ID ]
							);
							if( $attach && $attach = array_shift( $attach ) )
								$thumb_id = $attach->ID;
						}
					}
				}
			}
			elseif( $is_term ){
				if( ! $thumb_id = get_term_meta( $term->term_id, '_thumbnail_id', 1 ) ){
					$thumb_id = $fn__get_thumb_id_from_text( $term->description );
				}
			}

			/**
			 * Allow to set custom `og:image` data (URL).
			 *
			 * @param int|string|array  $thumb_id  Attachment ID. Image URL. Array of [ image_url, width, height ].
			 */
			$thumb_id = apply_filters( 'kama_og_meta_thumb_id', $thumb_id );

			if( $thumb_id ){

				if( is_numeric($thumb_id) )
					list( $image_url, $img_width, $img_height ) = image_downsize( $thumb_id, 'full' );
				elseif( is_array($thumb_id) )
					list( $image_url, $img_width, $img_height ) = $thumb_id;
				else
					$image_url = $thumb_id;

				// Open Graph image
				$els['og:image'] = '<meta property="og:image" content="'. esc_url($image_url) .'" />';
				if( isset($img_width) )
				$els['og:image:width']  = '<meta property="og:image:width" content="'. (int) $img_width .'" />';
				if( isset($img_height) )
				$els['og:image:height'] = '<meta property="og:image:height" content="'. (int) $img_height .'" />';
			}

		}

		// twitter
		$els['twitter:card']        = '<meta name="twitter:card" content="summary" />';
		$els['twitter:description'] = '<meta name="twitter:description" content="'. esc_attr( $desc ) .'" />';
		$els['twitter:title']       = '<meta name="twitter:title" content="'. esc_attr( $title ) .'" />';
		if( isset($image_url) )
		$els['twitter:image'] = '<meta name="twitter:image" content="'. esc_url($image_url) .'" />';

		/**
		 * Filter resulting properties. Allows to add or remove any og/twitter properties.
		 *
		 * @param array  $els
		 */
		$els = apply_filters( 'kama_og_meta_elements', $els );

		return "\n". implode("\n", $els ) ."\n\n";
	}

	/**
	 * Выводит заголовок страницы <title>
	 *
	 * Для меток и категорий указывается в настройках, в описании: [title=Заголовок].
	 * Для записей, если нужно, чтобы заголовок страницы отличался от заголовка записи,
	 * создайте произвольное поле title и впишите туда произвольный заголовок.
	 *
	 * @version 4.9
	 *
	 * @param string     $sep            разделитель
	 * @param true|false $add_blog_name  добавлять ли название блога в конец заголовка для архивов.
	 */
	static function meta_title( $sep = '»', $add_blog_name = true ){
		static $cache; if( $cache ) return $cache;

		global $post;

		$l10n = apply_filters( 'kama_meta_title_l10n', array(
			'404'     => 'Ошибка 404: такой страницы не существует',
			'search'  => 'Результаты поиска по запросу: %s',
			'compage' => 'Комментарии %s',
			'author'  => 'Статьи автора: %s',
			'archive' => 'Архив за',
			'paged'   => '(страница %d)',
		) );

		$parts = array(
			'prev'  => '',
			'title' => '',
			'after' => '',
			'paged' => '',
		);
		$title = & $parts['title']; // упростим
		$after = & $parts['after']; // упростим

		if(0){}
		// 404
		elseif ( is_404() ){
			$title = $l10n['404'];
		}
		// поиск
		elseif ( is_search() ){
			$title = sprintf( $l10n['search'], get_query_var('s') );
		}
		// главная
		elseif( is_front_page() ){
			if( is_page() && $title = get_post_meta( $post->ID, 'title', 1 ) ){
				// $title определен
			} else {
				$title = get_bloginfo('name');
				$after = get_bloginfo('description');
			}
		}
		// отдельная страница
		elseif( is_singular() || ( is_home() && ! is_front_page() ) || ( is_page() && ! is_front_page() ) ){
			$title = get_post_meta( $post->ID, 'title', 1 ); // указанный title у записи в приоритете

			if( ! $title ) $title = apply_filters( 'kama_meta_title_singular', '', $post );
			if( ! $title ) $title = single_post_title( '', 0 );

			if( $cpage = get_query_var('cpage') )
				$parts['prev'] = sprintf( $l10n['compage'], $cpage );
		}
		// архив типа поста
		elseif ( is_post_type_archive() ){
			$title = post_type_archive_title('', 0 );
			$after = 'blog_name';
		}
		// таксономии
		elseif( is_category() || is_tag() || is_tax() ){
			$term = get_queried_object();

			$title = get_term_meta( $term->term_id, 'title', 1 );

			if( ! $title ){
				$title = single_term_title('', 0 );

				if( is_tax() )
					$parts['prev'] = get_taxonomy($term->taxonomy)->labels->name;
			}

			$after = 'blog_name';
		}
		// архив автора
		elseif ( is_author() ){
			$title = sprintf( $l10n['author'], get_queried_object()->display_name );
			$after = 'blog_name';
		}
		// архив даты
		elseif ( ( get_locale() === 'ru_RU' ) && ( is_day() || is_month() || is_year() ) ){
			$rus_month  = array('', 'январь', 'февраль', 'март', 'апрель', 'май', 'июнь', 'июль', 'август', 'сентябрь', 'октябрь', 'ноябрь', 'декабрь');
			$rus_month2 = array('', 'января', 'февраля', 'марта', 'апреля', 'мая', 'июня', 'июля', 'августа', 'сентября', 'октября', 'ноября', 'декабря');
			$year       = get_query_var('year');
			$monthnum   = get_query_var('monthnum');
			$day        = get_query_var('day');

			if( is_year() )      $dat = "$year год";
			elseif( is_month() ) $dat = "$rus_month[$monthnum] $year года";
			elseif( is_day() )   $dat = "$day $rus_month2[$monthnum] $year года";

			$title = sprintf( $l10n['archive'], $dat );
			$after = 'blog_name';
		}
		// остальные архивы
		else {
			$title = get_the_archive_title();
			$after = 'blog_name';
		}

		// номера страниц для пагинации и деления записи
		$pagenum = get_query_var('paged') ?: get_query_var('page');
		if( $pagenum )
			$parts['paged'] = sprintf( $l10n['paged'], $pagenum );

		// позволяет фильтровать title как угодно. Сам заголово
		// $parts содержит массив с элементами: prev - текст до, title - заголовок, after - текст после
		$parts = apply_filters_ref_array( 'kama_meta_title_parts', array($parts, $l10n) );

		if( $after == 'blog_name' )
			$after = $add_blog_name ? get_bloginfo('name') : '';

		// добавим пагинацию в title
		if( $parts['paged'] ){
			$parts['title'] .=  " {$parts['paged']}";
			unset( $parts['paged'] );
		}

		$title = implode( ' '. trim($sep) .' ', array_filter($parts) );

		$title = apply_filters( 'kama_meta_title', $title );

		$title = wptexturize( $title );
		$title = esc_html( $title );

		return $cache = $title;
	}

	/**
	 * Выводит метатег description.
	 *
	 * Для элементов таксономий: метаполе description или в описании такой шоткод [description = текст описания]
	 * У постов сначала проверяется, метаполе description, или цитата, или начальная часть контента.
	 * Цитата или контент обрезаются до указанного в $maxchar символов.
	 *
	 * @param string $home_description Указывается описание для главной страницы сайта.
	 * @param int    $maxchar          Максимальная длина описания (в символах).
	 *
	 * @version 2.2.2
	 */
	static function meta_description( $home_description = '', $maxchar = 260 ){
		static $cache; if( $cache ) return $cache;

		global $post;

		$cut   = true;
		$desc  = '';

		// front
		if( is_front_page() ){
			// когда для главной установлена страница
			if( is_page() && $desc = get_post_meta($post->ID, 'description', true )  ){
				$cut = false;
			}

			if( ! $desc )
				$desc = $home_description ?: get_bloginfo( 'description', 'display' );
		}
		// singular
		elseif( is_singular() ){
			if( $desc = get_post_meta($post->ID, 'description', true ) )
				$cut = false;

			if( ! $desc ) $desc = $post->post_excerpt ?: $post->post_content;

			$desc = trim( strip_tags( $desc ) );
		}
		// term
		elseif( is_category() || is_tag() || is_tax() ){
			$term = get_queried_object();

			$desc = get_term_meta( $term->term_id, 'meta_description', true );
			if( ! $desc )
				$desc = get_term_meta( $term->term_id, 'description', true );

			$cut = false;
			if( ! $desc && $term->description ){
				$desc = strip_tags( $term->description );
				$cut = true;
			}
		}

		$origin_desc = $desc;

		if( $desc = apply_filters( 'kama_meta_description_pre', $desc ) ){

			$desc = str_replace( array("\n", "\r"), ' ', $desc );
			$desc = preg_replace( '~\[[^\]]+\](?!\()~', '', $desc ); // удаляем шоткоды. Оставляем маркдаун [foo](URL)

			if( $cut ){
				$char = mb_strlen( $desc );
				if( $char > $maxchar ){
					$desc     = mb_substr( $desc, 0, $maxchar );
					$words    = explode(' ', $desc );
					$maxwords = count($words) - 1; // убираем последнее слово, оно в 90% случаев неполное
					$desc     = join(' ', array_slice($words, 0, $maxwords)).' ...';
				}
			}

			$desc = preg_replace( '/\s+/s', ' ', $desc );
		}

		if( $desc = apply_filters( 'kama_meta_description', $desc, $origin_desc, $cut, $maxchar ) )
		return $cache = '<meta name="description" content="'. esc_attr( trim($desc) ) .'" />'."\n";

		return $cache = '';
	}

	/**
	 * Метатег robots
	 *
	 * Чтобы задать свои атрибуты метатега robots записи, создайте произвольное поле с ключом robots
	 * и необходимым значением, например: noindex,nofollow
	 *
	 * Укажите параметр $allow_types, чтобы разрешить индексацию типов страниц.
	 *
	 * @ $allow_types Какие типы страниц нужно индексировать (через запятую):
	 *                cpage, is_category, is_tag, is_tax, is_author, is_year, is_month,
	 *                is_attachment, is_day, is_search, is_feed, is_post_type_archive, is_paged
	 *                (можно использовать любые условные теги в виде строки)
	 *                cpage - страницы комментариев
	 * @ $robots      Как закрывать индексацию: noindex,nofollow
	 *
	 * version 0.8
	 */
	static function meta_robots( $allow_types = null, $robots = 'noindex,nofollow' ){
		global $post;

		if( null === $allow_types )
			$allow_types = 'cpage, is_attachment, is_category, is_tag, is_tax, is_paged, is_post_type_archive';

		if( ( is_home() || is_front_page() ) && ! is_paged() )
			return '';

		if( is_singular() ){
			// если это не вложение или вложение но оно разрешено
			if( ! is_attachment() || false !== strpos($allow_types,'is_attachment') )
				$robots = get_post_meta( $post->ID, 'robots', true );
		}
		else {
			$types = preg_split('~[, ]+~', $allow_types );
			$types = array_filter( $types );

			foreach( $types as $type ){
				if( $type == 'cpage' && strpos($_SERVER['REQUEST_URI'], '/comment-page') ) $robots = false;
				elseif( function_exists($type) && $type() )                                $robots = false;
			}
		}

		$robots = apply_filters( 'kama_meta_robots_close', $robots );

		return $robots ? "<meta name=\"robots\" content=\"$robots\" />\n" : '';
	}

	/**
	 * Генерирует метатег keywords для head части старницы.
	 *
	 * Чтобы задать свои keywords для записи, создайте произвольное поле keywords и впишите в значения необходимые ключевые слова.
	 * Для постов (post) ключевые слова генерируются из меток и названия категорий, если не указано произвольное поле keywords.
	 *
	 * Для меток, категорий и произвольных таксономий, ключевые слова указываются в описании, в шоткоде: [ keywords=слово1, слово2, слово3 ]
	 *
	 * @ $home_keywords: Для главной, ключевые слова указываются в первом параметре: meta_keywords( 'слово1, слово2, слово3' );
	 * @ $def_keywords: сквозные ключевые слова - укажем и они будут прибавляться к остальным на всех страницах
	 *
	 * version 0.8
	 */
	static function meta_keywords( $home_keywords = '', $def_keywords = '' ){
		global $post;

		$out = '';

		if ( is_front_page() ){
			$out = $home_keywords;
		}
		elseif( is_singular() ){
			$out = get_post_meta( $post->ID, 'keywords', true );

			// для постов указываем ключами метки и категории, если не указаны ключи в произвольном поле
			if( ! $out && $post->post_type == 'post' ){
				$res = wp_get_object_terms( $post->ID, [ 'post_tag', 'category' ], [ 'orderby' => 'none' ] ); // получаем категории и метки

				if( $res && ! is_wp_error($res) )
					foreach( $res as $tag )
						$out .= ", $tag->name";

				$out = ltrim( $out, ', ' );
			}
		}
		elseif ( is_category() || is_tag() || is_tax() ){
			$term = get_queried_object();

			// wp 4.4
			if( function_exists('get_term_meta') ){
				$out = get_term_meta( $term->term_id, "keywords", true );
			}
			else{
				preg_match( '!\[keywords=([^\]]+)\]!iU', $term->description, $match );
				$out = isset($match[1]) ? $match[1] : '';
			}
		}

		if( $out && $def_keywords )
			$out = $out .', '. $def_keywords;

		return $out ? "<meta name=\"keywords\" content=\"$out\" />\n" : '';
	}

}
solass_wp_SEO_Tags::init();
?>
<?php

if( ! class_exists('solass_wp_Post_Meta_Box') ) :

	/**
	 * Создает блок произвольных полей для указанных типов записей.
	 *
	 * Возможные параметры класса, смотрите в: solass_wp_Post_Meta_Box::__construct()
	 * Возможные параметры для каждого поля, смотрите в: solass_wp_Post_Meta_Box::field()
	 *
	 * При сохранении, очищает каждое поле, через: wp_kses() или sanitize_text_field().
	 * Функцию очистки можно заменить через хук 'kpmb_save_sanitize_{$id}' и
	 * также можно указать название функции очистки в параметре 'save_sanitize'.
	 * Если указать функции очистки и в параметре, и в хуке, то будут работать обе!
	 * Обе функции очистки получают данные: $metas - все метаполя, $post_id - ID записи.
	 *
	 * Блок выводиться и метаполя сохраняются для юзеров с правом редактировать текущую запись.
	 *
	 * PHP: 5.3+
	 *
	 * @changlog https://github.com/doiftrue/Kama_Post_Meta_Box/blob/master/changelog.md
	 *
	 * @version 1.9.7
	 */
	class solass_wp_Post_Meta_Box {

		public $opt;

		public $id;

		static $instances = array(); // сохраняет ссылки на все экземпляры

		/**
		 * Конструктор
		 *
		 * @param array $opt Опции по которым будет строиться метаблок
		 */
		function __construct( $opt ){

			$defaults = [
				'id'         => '',       // динетификатор блока. Используется как префикс для названия метаполя.
				// начните идент. с '_': '_foo', чтобы ID не был префиксом в названии метаполей.

				'title'      => '',       // заголовок блока
				'desc'       => '',       // описание для метабокса. Можно указать функцию/замыкание, она получит $post. С версии 1.9.1
				'post_type'  => '',       // строка/массив. Типы записей для которых добавляется блок: array('post','page').
				// По умолчанию: '' - для всех типов записей.
				'post_type_feature' => '', // строка. возможность которая должна быть у типа записи,
				// чтобы метабокс отобразился. См. post_type_supports()
				'post_type_options' => '', // массив. опции типа записи, которые должны быть у типа записи чтобы метабокс отобразился.
				// см. первый параметр get_post_types()

				'priority'   => 'high',   // Приоритет блока для показа выше или ниже остальных блоков ('high' или 'low').
				'context'    => 'normal', // Место где должен показываться блок ('normal', 'advanced' или 'side').

				'disable_func'  => '',    // функция отключения метабокса во время вызова самого метабокса.
				// Если вернет что-либо кроме false/null/0/array(), то метабокс будет отключен. Передает объект поста.

				'cap'           => '',    // название права пользователя, чтобы показывать метабокс.

				'save_sanitize' => '',    // Функция очистки сохраняемых в БД полей. Получает 2 параметра:
				// $metas - все поля для очистки и $post_id

				'theme' => 'table',       // тема оформления: 'table', 'line'.
				// ИЛИ массив паттернов полей: css, fields_wrap, field_wrap, title_patt, field_patt, desc_patt.
				// Массив указывается так: [ 'desc_patt' => '<div>%s</div>' ] (за овнову будет взята тема line)
				// Массив указывается так: [ 'table' => [ 'desc_patt' => '<div>%s</div>' ] ] (за овнову будет взята тема table)
				// ИЛИ изменить тему можно через фильтр 'kp_metabox_theme' - удобен для общего изменения темы для всех метабоксов.

				// метаполя. Параметры смотрите ниже в методе field()
				'fields'     => [
					'foo' => [ 'title' =>'Первое метаполе' ],
					'bar' => [ 'title' =>'Второе метаполе' ],
				],
			];

			$this->opt = (object) array_merge( $defaults, $opt );

			// хуки инициализации, вешается на хук init чтобы текущий пользователь уже был установлен
			add_action( 'init', [ $this, 'init_hooks' ], 20 );
		}

		function init_hooks(){
			// может метабокс отключен по праву
			if( $this->opt->cap && ! current_user_can( $this->opt->cap ) )
				return;

			// темы оформления
			if( 'theme_options' ){

				$opt_theme = & $this->opt->theme;

				$def_opt_theme = [
					'line' => [
						// CSS стили всего блока. Например: '.postbox .tit{ font-weight:bold; }'
						'css'         => '',
						// '%s' будет заменено на html всех полей
						'fields_wrap' => '%s',
						// '%2$s' будет заменено на html поля (вместе с заголовком, полем и описанием)
						'field_wrap'  => '<p class="%1$s">%2$s</p>',
						// '%s' будет заменено на заголовок
						'title_patt'  => '<strong class="tit"><label>%s</label></strong>',
						// '%s' будет заменено на HTML поля (вместе с описанием)
						'field_patt'  => '%s',
						// '%s' будет заменено на текст описания
						'desc_patt'   => '<br><span class="description" style="opacity:0.6;">%s</span>',
					],
					'table' => [
						'css'         => '.kpmb-table td{ padding:.6em .5em; } .kpmb-table tr:hover{ background:rgba(0,0,0,.03); }',
						'fields_wrap' => '<table class="form-table kpmb-table">%s</table>',
						'field_wrap'  => '<tr class="%1$s">%2$s</tr>',
						'title_patt'  => '<td style="width:10em;" class="tit">%s</td>',
						'field_patt'  => '<td class="field">%s</td>',
						'desc_patt'   => '<br><span class="description" style="opacity:0.8;">%s</span>',
					],
				];

				if( is_string($opt_theme) ){
					$def_opt_theme = $def_opt_theme[ $opt_theme ];
				}
				// позволяет изменить отдельные поля темы оформелния метабокса
				else {
					$opt_theme_key = key( $opt_theme ); // индекс массива

					// в индексе указана не тема: [ 'desc_patt' => '<div>%s</div>' ]
					if( ! in_array( $opt_theme_key, array_keys($def_opt_theme) ) ){
						$def_opt_theme = $def_opt_theme['line']; // основа темы
					}
					// в индексе указана тема: [ 'table' => [ 'desc_patt' => '<div>%s</div>' ] ]
					else {
						$def_opt_theme = $def_opt_theme[ $opt_theme_key ]; // основа темы
						$opt_theme     = $opt_theme[ $opt_theme_key ];
					}
				}

				$opt_theme = is_array( $opt_theme ) ? array_merge( $def_opt_theme, $opt_theme ) : $def_opt_theme;

				// для изменения темы через фильтр
				$opt_theme = apply_filters( 'kp_metabox_theme', $opt_theme, $this->opt );

				// переменные theme в общие параметры.
				// Если в параметрах уже есть переменная, то она остается как есть (это позволяет изменить отдельный элемент темы).
				foreach( $opt_theme as $kk => $vv ){
					if( ! isset($this->opt->{$kk}) )
						$this->opt->{$kk} = $vv;
				}
			}

			// создадим уникальный ID объекта
			$_opt = (array) clone $this->opt;
			// удалим (очистим) все closure
			array_walk_recursive( $_opt, function(&$val, $key){
				if( $val instanceof Closure ) $val = '';
			});
			$this->id = substr( md5(serialize($_opt)), 0, 7 ); // ID экземпляра

			// сохраним ссылку на экземпляр, чтобы к нему был доступ
			self::$instances[ $this->opt->id ][ $this->id ] = & $this;

			add_action( 'add_meta_boxes', [ $this, 'add_meta_box' ], 10, 2 );
			add_action( 'save_post', [ $this, 'meta_box_save' ], 1, 2 );
		}

		function add_meta_box( $post_type, $post ){

			$opt = $this->opt; // short love

			// может отключить метабокс?
			if(
				in_array( $post_type, [ 'comment','link' ] )
				|| ! current_user_can( get_post_type_object( $post_type )->cap->edit_post, $post->ID )
				|| ( $opt->post_type_feature && ! post_type_supports( $post_type, $opt->post_type_feature ) )
				|| ( $opt->post_type_options && ! in_array( $post_type, get_post_types( $opt->post_type_options, 'names', 'or' ) ) )
				|| ( $opt->disable_func && is_callable($opt->disable_func) && call_user_func( $opt->disable_func, $post ) )
			)
				return;

			$p_types = $opt->post_type ?: $post_type;

			// if WP < 4.4
			if( is_array($p_types) && version_compare( $GLOBALS['wp_version'], '4.4', '<' ) ){
				foreach( $p_types as $p_type )
					add_meta_box( $this->id, $opt->title, [ $this, 'meta_box' ], $p_type, $opt->context, $opt->priority );
			}
			else
				add_meta_box( $this->id, $opt->title, [ $this, 'meta_box' ], $p_types, $opt->context, $opt->priority );

			// добавим css класс к метабоксу
			// apply_filters( "postbox_classes_{$page}_{$id}", $classes );
			add_filter( "postbox_classes_{$post_type}_{$this->id}", [ $this, '_postbox_classes_add' ] );
		}

		/**
		 * Выводит код блока
		 * @param object $post Объект записи
		 */
		function meta_box( $post ){
			$fields_out = $hidden_out = '';

			foreach( $this->opt->fields as $key => $args ){
				if( ! $key || ! $args ) continue; // пустое поле

				if( empty($args['title_patt']) ) $args['title_patt'] = @ $this->opt->title_patt ?: '%s';
				if( empty($args['desc_patt'])  ) $args['desc_patt']  = @ $this->opt->desc_patt  ?: '%s';
				if( empty($args['field_patt']) ) $args['field_patt'] = @ $this->opt->field_patt ?: '%s';

				$args['key'] = $key;

				$field_wrap = & $this->opt->field_wrap;
				if( @ $args['type'] == 'wp_editor' )  $field_wrap = str_replace( [ '<p ','</p>' ], [ '<div ','</div><br>' ], $field_wrap );

				if( @ $args['type'] == 'hidden' )
					$hidden_out .= $this->field( $args, $post );
				else
					$fields_out .= sprintf( $field_wrap, $key .'_meta', $this->field( $args, $post ) );

			}

			$metabox_desc = '';
			if( $this->opt->desc )
				$metabox_desc = is_callable($this->opt->desc) ? call_user_func($this->opt->desc, $post) : '<p class="description">'. $this->opt->desc .'</p>';

			echo ( $this->opt->css ? '<style>'. $this->opt->css .'</style>' : '' ) .
			     $metabox_desc .
			     $hidden_out .
			     sprintf( (@ $this->opt->fields_wrap ?: '%s'), $fields_out ) .
			     '<div class="clear"></div>';
		}

		/**
		 * Выводит отдельные мета поля.
		 *
		 * @param string  $name  Атрибут name.
		 * @param array   $args  Параметры поля.
		 * @param object  $post  Объект текущего поста.
		 *
		 * @return string|null HTML код
		 */
		function field( $args, $post ){

			$var = (object) []; // внутренние переменные этой фукнции, будут переданы в методы
			$rg  = (object) array_merge( [
				'type'          => '', // тип поля: textarea, select, checkbox, radio, image, wp_editor, hidden, sep_*.
				// Или базовые: text, email, number, url, tel, color, password, date, month, week, range.
				// 'sep' - визуальный разделитель, для него нужно указать `title` и можно указать `'attr'=>'style="свои стили"'`.
				// 'sep' - чтобы удобнее указывать тип 'sep' начните ключ поля с `sep_`: 'sep_1' => [ 'title'=>'Разделитель' ].
				// Для типа `image` можно указать тип сохраняемого значения в `options`: 'options'=>'url'. По умолчанию тип = id.
				// По умолчанию 'text'.

				'title'         => '', // заголовок метаполя
				'desc'          => '', // описание для поля. Можно указать функцию/замыкание, она получит параметры: $post, $meta_key, $val, $name.
				'placeholder'   => '', // атрибут placeholder
				'id'            => '', // атрибут id. По умолчанию: $this->opt->id .'_'. $key
				'class'         => '', // атрибут class: добавляется в input, textarea, select. Для checkbox, radio в оборачивающий label
				'attr'          => '', // любая строка, будет расположена внутри тега. Для создания атрибутов. Пр: style="width:100%;"
				'val'           => '', // значение по умолчанию, если нет сохраненного.
				'options'       => '', // массив: array('значение'=>'название') - варианты для типов 'select', 'radio'.
				// Для 'wp_editor' стенет аргументами.
				// Для 'checkbox' станет значением атрибута value: <input type="checkbox" value="{options}">.
				// Для 'image' определяет тип сохраняемого в метаполе значения: id (ID вложения), url (url вложения).

				'callback'      => '', // название функции, которая отвечает за вывод поля.
				// если указана, то ни один параметр не учитывается и за вывод полностью отвечает указанная функция.
				// Все параметры передаются ей... Получит параметры: $args, $post, $name, $val

				'sanitize_func' => '', // функция очистки данных при сохранении - название функции или Closure.
				// Укажите 'none', чтобы не очищать данные...
				// работает, только если не установлен глобальный параметр 'save_sanitize'...
				// получит параметр $value - сохраняемое значение поля.

				'output_func'   => '', // функция обработки значения, перед выводом в поле.
				// получит параметры: $post, $meta_key, $value - объект записи, ключ, значение метаполей.

				'update_func'   => '', // функция сохранения значения в метаполя.
				// получит параметры: $post, $meta_key, $value - объект записи, ключ, значение метаполей.

				'disable_func'  => '', // функция отключения поля.
				// Если не false/null/0/array() - что-либо вернет, то поле не будет выведено.
				// Получает парамтры: $post, $meta_key

				'cap'           => '', // название права пользователя, чтобы видеть и изменять поле.

				// служебные
				'key'           => '', // Обязательный! Автоматический
				'title_patt'    => '', // Обязательный! Автоматический
				'field_patt'    => '', // Обязательный! Автоматический
				'desc_patt'     => '', // Обязательный! Автоматический
			], $args );

			if( $rg->cap && ! current_user_can( $rg->cap ) )
				return null;

			if( 'sep_' === substr($rg->key, 0, 4) ) $rg->type = 'sep';
			if( ! $rg->type )                       $rg->type = 'text';

			$var->meta_key = $this->_key_prefix() . $rg->key;

			// поле отключено
			if( $rg->disable_func && is_callable($rg->disable_func) && call_user_func( $rg->disable_func, $post, $var->meta_key ) )
				return null;

			// meta_val
			$rg->val = get_post_meta( $post->ID, $var->meta_key, true ) ?: $rg->val;
			if( $rg->output_func && is_callable($rg->output_func) )
				$rg->val = call_user_func( $rg->output_func, $post, $var->meta_key, $rg->val );

			$var->name = $this->id . "_meta[$var->meta_key]";

			$rg->id  = $rg->id ?: ( $this->opt->id .'_'. $rg->key );

			// при табличной теме, td заголовка должен выводиться всегда!
			if( false !== strpos($rg->title_patt, '<td ') )
				$var->title = sprintf( $rg->title_patt, $rg->title ) . ($rg->title ? ' ' : '');
			else
				$var->title = $rg->title ? sprintf( $rg->title_patt, $rg->title ) .' ' : '';

			$rg->options = (array) $rg->options;

			$var->pholder = $rg->placeholder ? ' placeholder="'. esc_attr($rg->placeholder) .'"' : '';
			$var->class = $rg->class ? ' class="'. $rg->class .'"' : '';

			$fn__desc = function() use ( $rg, $post, $var ){
				if( ! $rg->desc ) return '';
				$desc = is_callable( $rg->desc ) ? call_user_func_array($rg->desc, [ $post, $var->meta_key, $rg->val, $var->name ] ) : $rg->desc;
				return sprintf( $rg->desc_patt, $desc );
			};

			$fn__field = function( $field ) use ( $rg ){
				return sprintf( $rg->field_patt, $field );
			};

			// произвольная функция
			if( is_callable( $rg->callback ) )
				$out = $var->title . $fn__field( call_user_func_array( $rg->callback, [ $args, $post, $var->name, $rg->val, $rg, $var ] ) );
			// произвольный метод
			// вызов метода `$this->field__{FIELD}()` (для возможности расширить этот класс)
			elseif( method_exists( $this, "field__$rg->type") )
				$out = $this->{"field__$rg->type"}( $rg, $var, $post, $fn__desc, $fn__field );
			// text, email, number, url, tel, color, password, date, month, week, range
			else
				$out = $this->field__default( $rg, $var, $post, $fn__desc, $fn__field );

			return $out;
		}

		// textarea
		function field__textarea( $rg, $var, $post, $fn__desc, $fn__field ){
			$_style = (false === strpos($rg->attr,'style=')) ? ' style="width:98%;"' : '';

			return $var->title . $fn__field('<textarea '. $rg->attr . $var->class . $var->pholder . $_style .'  id="'. $rg->id .'" name="'. $var->name .'">'. esc_textarea($rg->val) .'</textarea>'. $fn__desc() );
		}

		// select
		function field__select( $rg, $var, $post, $fn__desc, $fn__field ){
			$is_assoc = ( array_keys($rg->options) !== range(0, count($rg->options) - 1) ); // ассоциативный или нет?
			$_options = array();
			foreach( $rg->options as $v => $l ){
				$_val       = $is_assoc ? $v : $l;
				$_options[] = '<option value="'. esc_attr($_val) .'" '. selected($rg->val, $_val, 0) .'>'. $l .'</option>';
			}

			return $var->title . $fn__field('<select '. $rg->attr . $var->class .' id="'. $rg->id .'" name="'. $var->name .'">' . implode("\n", $_options ) . '</select>' . $fn__desc() );
		}

		// radio
		function field__radio( $rg, $var, $post, $fn__desc, $fn__field ){
			$radios = array();
			foreach( $rg->options as $v => $l )
				$radios[] = '<label '. $rg->attr . $var->class .'><input type="radio" name="'. $var->name .'" value="'. $v .'" '. checked($rg->val, $v, 0) .'>'. $l .'</label> ';

			return $var->title . $fn__field('<span class="radios">'. implode("\n", $radios ) .'</span>'. $fn__desc() );
		}

		// checkbox
		function field__checkbox( $rg, $var, $post, $fn__desc, $fn__field ){
			return $var->title . $fn__field('
				<label '. $rg->attr . $var->class .'>
					<input type="hidden" name="'. $var->name .'" value="">
					<input type="checkbox" id="'. $rg->id .'" name="'. $var->name .'" value="'. esc_attr(reset($rg->options) ?: 1) .'" '. checked( $rg->val, (reset($rg->options) ?: 1), 0) .'>
					'.( $rg->desc ?: '' ).'
				</label>');
		}

		// sep
		function field__sep( $rg, $var, $post, $fn__desc, $fn__field ){
			$_style = 'font-weight:600; ';
			if( preg_match( '/style="([^"]+)"/', $rg->attr, $mm ) ) $_style .= $mm[1];

			if( false !== strpos( $rg->field_patt, '<td' ) )
				return str_replace( '<td ', '<td colspan="2" style="padding:1em .5em; '. $_style .'"', $fn__field( $rg->title ) );
			else
				return '<span style="display:block; padding:1em 0; font-size:110%; '. $_style .'">'. $rg->title .'</span>';
		}

		// hidden
		function field__hidden( $rg, $var, $post, $fn__desc, $fn__field ){
			return '<input type="'. $rg->type .'" id="'. $rg->id .'" name="'. $var->name .'" value="'. esc_attr($rg->val) .'" title="'. esc_attr($rg->title) .'">';
		}

		// text, email, number, url, tel, color, password, date, month, week, range
		function field__default( $rg, $var, $post, $fn__desc, $fn__field ){
			$_style   = ( $rg->type === 'text' && false === strpos($rg->attr, 'style=') ) ? ' style="width:100%;"' : '';

			return $var->title . $fn__field( '<input '. $rg->attr . $var->class  . $var->pholder . $_style .' type="'. $rg->type .'" id="'. $rg->id .'" name="'. $var->name .'" value="'. esc_attr($rg->val) .'">'. $fn__desc() );
		}

		// wp_editor
		function field__wp_editor( $rg, $var, $post, $fn__desc, $fn__field ){
			$ed_args = array_merge( [
				'textarea_name'    => $var->name, //нужно указывать!
				'editor_class'     => $rg->class,
				// изменяемое
				'wpautop'          => 1,
				'textarea_rows'    => 5,
				'tabindex'         => null,
				'editor_css'       => '',
				'teeny'            => 0,
				'dfw'              => 0,
				'tinymce'          => 1,
				'quicktags'        => 1,
				'media_buttons'    => false,
				'drag_drop_upload' => false,
			], $rg->options );

			ob_start();
			wp_editor( $rg->val, $rg->id, $ed_args );
			$wp_editor = ob_get_clean();

			return $var->title . $fn__field( $wp_editor . $fn__desc() );
		}

		// image
		function field__image( $rg, $var, $post, $fn__desc, $fn__field ){

			wp_enqueue_media();

			static $once;
			if( ! $once && $once = 1 ){
				add_action( 'admin_print_footer_scripts', function(){
					?>
					<script>
					$('.kmb_img_wrap').each(function(){

						var frame,
							$wrap = $(this),
							$img   = $wrap.find('img'),
							$input = $wrap.find('input[type="hidden"]');

						$wrap.on( 'click', '.set_img', function(){

							var post_id = $(this).data('post_id') || null

							//if( frame && frame.post_id === post_id ){
							//	frame.open();
							//	return;
							//}

							frame = wp.media.frames.kmbframe = wp.media({
								title   : '<?= __( 'Add Media' ) ?>',
								// Library WordPress query arguments.
								library : {
									type       : 'image',
									uploadedTo : post_id
								},
								multiple: false,
								button: {
									text: '<?= __( 'Apply' ) ?>'
								}
							});

							frame.on( 'select', function() {
								attachment = frame.state().get('selection').first().toJSON();
								$img.attr( 'src', attachment.url );

								$wrap.data('usetype') === 'url' ? $input.val( attachment.url ) : $input.val( attachment.id );
							});

							frame.on( 'open', function(){
								if( $input.val() )
									frame.state().get('selection').add( wp.media.attachment( $input.val() ) );
							});

							frame.open();
							//frame.post_id = post_id // save
						});

						$wrap.on( 'click', '.del_img', function(){
							$img.attr( 'src', '' );
							$input.val('');
						});
					})
					</script>
					<?php
				}, 99 );
			}

			$usetype = $rg->options ? $rg->options[0] : 'id'; // может быть: id, url

			if( ! $src = is_numeric($rg->val) ? wp_get_attachment_url( $rg->val ) : $rg->val )
				$src = 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAQAAAC1HAwCAAAAC0lEQVR42mNkYAAAAAYAAjCB0C8AAAAASUVORK5CYII=';

			ob_start();
			?>
			<span class="kmb_img_wrap" data-usetype="<?= esc_attr($usetype) ?>" style="display:flex; align-items:center;">
				<img src="<?= esc_url($src) ?>" style="max-height:100px; max-width:100px; margin-right:1em;" alt="">
				<span>
					<input class="set_img button button-small" type="button" value="<?= __('Set image') ?>" />
					<input class="set_img button button-small" type="button" data-post_id="<?= $post->ID ?>" value="<?= __( 'Uploaded to this post' ) ?>" />
					<input class="del_img button button-small" type="button" value="<?= __('Remove')?>" />

					<input type="hidden" name="<?= $var->name ?>" value="<?= esc_attr($rg->val) ?>">
				</span>
			</span>
			<?php
			$field = ob_get_clean();

			return $var->title . $fn__field( $field );
		}

		/**
		 * Сохраняем данные, при сохранении поста
		 * @param  integer $post_id ID записи
		 * @return boolean  false если проверка не пройдена
		 */
		function meta_box_save( $post_id, $post ){

			if(	! ( $save_metadata = @ $_POST["{$this->id}_meta"] )                                        // нет данных
			       || ( defined('DOING_AUTOSAVE') && DOING_AUTOSAVE  )                                           // выходим, если это автосохр.
			       || ! wp_verify_nonce( $_POST['_wpnonce'], 'update-post_'. $post_id )                          // nonce проверка
			       || ( $this->opt->post_type && ! in_array( $post->post_type, (array) $this->opt->post_type ) ) // не подходящий тип записи
			)
				return null;

			// оставим только поля текущего класса (защиты от подмены поля)
			$_key_prefix = $this->_key_prefix();
			$fields_data = array();
			foreach( $this->opt->fields as $_key => $rg ){
				$meta_key = $_key_prefix . $_key;

				// недостаточно прав
				if( !empty($rg['cap']) && ! current_user_can( $rg['cap'] ) )
					continue;

				// пропускаем отключенные поля
				if( !empty($rg['disable_func']) && is_callable($rg['disable_func']) && call_user_func( $rg['disable_func'], $post, $meta_key ) )
					continue;

				$fields_data[ $meta_key ] = $rg;
			}
			$fields_names  = array_keys( $fields_data );
			$save_metadata = array_intersect_key( $save_metadata, array_flip($fields_names) );


			// Очистка
			if( 'sanitize' ){
				// своя функция очистки
				if( is_callable($this->opt->save_sanitize) ){
					$save_metadata = call_user_func_array( $this->opt->save_sanitize, [ $save_metadata, $post_id, $fields_data ] );
					$sanitized = true;
				}
				// хук очистки
				if( has_filter("kpmb_save_sanitize_{$this->opt->id}") ){
					$save_metadata = apply_filters("kpmb_save_sanitize_{$this->opt->id}", $save_metadata, $post_id, $fields_data );
					$sanitized = true;
				}
				// если нет функции и хука очистки, то чистим все поля с помощью wp_kses() или sanitize_text_field()
				if( empty($sanitized) ){

					foreach( $save_metadata as $meta_key => & $value ){
						// есть функция очистки отдельного поля
						if( !empty($fields_data[$meta_key]['sanitize_func']) && is_callable($fields_data[$meta_key]['sanitize_func']) ){
							$value = call_user_func( $fields_data[$meta_key]['sanitize_func'], $value );
						}
						// не чистим
						elseif( @ $fields_data[$meta_key]['sanitize_func'] === 'none' ){}
						// не чистим - видимо это произвольная функция вывода полей, которая сохраняет массив
						elseif( is_array($value) ){}
						// нет функции очистки отдельного поля
						else {

							$type = !empty($fields_data[$meta_key]['type']) ? $fields_data[$meta_key]['type'] : 'text';

							if(0){}
							elseif( $type === 'number' )
								$value = floatval( $value );
							elseif( $type === 'email' )
								$value = sanitize_email( $value );
							// wp_editor, textarea
							elseif( in_array( $type, [ 'wp_editor','textarea' ], true ) )
								$value = addslashes( wp_kses( stripslashes( $value ), 'post' ) ); // default ?
							// text, radio, checkbox, color, date, month, tel, time, url
							else
								$value = sanitize_text_field( $value );
						}
					}
					unset($value); // $value используется ниже, поэтому он должен быть пустой, а не ссылкой...

				}
			}

			// Сохраняем
			foreach( $save_metadata as $meta_key => $value ){
				// если есть функция сохранения
				if( !empty($fields_data[$meta_key]['update_func']) && is_callable($fields_data[$meta_key]['update_func']) ){
					call_user_func( $fields_data[$meta_key]['update_func'], $post, $meta_key, $value );
				}
				else {
					// удаляем поле, если значение пустое. 0 остается...
					if( ! $value && ($value !== '0') )
						delete_post_meta( $post_id, $meta_key );
					else
						update_post_meta( $post_id, $meta_key, $value ); // add_post_meta() работает автоматически
				}
			}
		}

		function _postbox_classes_add( $classes ){
			$classes[] = "kama_meta_box_{$this->opt->id}";
			return $classes;
		}

		function _key_prefix(){
			return ($this->opt->id{0} == '_') ? '' : $this->opt->id .'_';
		}

	}

endif;
?>